/************************************************** -*- tab-width:4 -*- *
*
* Enhanced JavaScript Mode for Lugaru Epsilon 14.06
* Supports ES2015+ (ES6+), JSX, and JSON
*
* Based on Epsilon's C mode with JavaScript-specific enhancements
* Created: 2025
************************************************************************/


#include "eel.h"
#include "c.h"
#include "kill.h"
#include "colcode.h"
#include "proc.h"
#include "tags.h"

// JavaScript-specific color classes
// Default to standard C classes so they work with any color scheme
color_class jsx_tag = color_class c_function;          // JSX tag names like <Component>
color_class jsx_attribute = color_class c_identifier;  // JSX attributes
color_class js_template_expr = color_class c_string;   // Template literal expressions ${...}
color_class js_regex = color_class c_string;           // Regular expression literals

// Minimal tracing helpers - writes to #js-debug# buffer
int TRACE_ENABLED = 0;   // Set to 1 to enable debug tracing

void trace(char *msg)
{
	if (!TRACE_ENABLED) return;
	int saved_buf = bufnum;
	to_buffer("#js-debug#");
	point = size();
	bprintf("%s\n", msg);
	bufnum = saved_buf;
}

void trace_i(char *label, int v)
{
	if (!TRACE_ENABLED) return;
	int saved_buf = bufnum;
	to_buffer("#js-debug#");
	point = size();
	bprintf("%s%d\n", label, v);
	bufnum = saved_buf;
}

void trace_2i(char *label, int v1, int v2)
{
	if (!TRACE_ENABLED) return;
	int saved_buf = bufnum;
	to_buffer("#js-debug#");
	point = size();
	bprintf("%s%d,%d\n", label, v1, v2);
	bufnum = saved_buf;
}

void trace_3i(char *label, int v1, int v2, int v3)
{
	if (!TRACE_ENABLED) return;
	int saved_buf = bufnum;
	to_buffer("#js-debug#");
	point = size();
	bprintf("%s%d,%d,%d\n", label, v1, v2, v3);
	bufnum = saved_buf;
}

void trace_enter(char *func)
{
	if (!TRACE_ENABLED) return;
	int saved_buf = bufnum;
	to_buffer("#js-debug#");
	point = size();
	bprintf(">> %s()\n", func);
	bufnum = saved_buf;
}

// Forward declarations
int color_js_es2015_range(int from, int to);
int do_color_js_es2015_range(int from, int to);
int color_json_range(int from, int to);
int js_keyword_color(int from);
int at_jsx_tag();
void color_jsx_tag(int start);
int is_js_es2015_keyword(char *p);
int color_narrowed_c_range(int start, int end);
void color_js_template_literal(int start);

// Enhanced JavaScript mode with ES2015+ and JSX support
command javascript_es2015_mode()
{
	c_mode();
	c_extra_keywords = JS_KEYWORDS;
	major_mode = "JavaScript";
	strcpy(comment_start, "//[ \t\f]*");
	strcpy(comment_pattern, "//.*$|/<*>(.|<newline>)*<*>/<FirstEnd>");

	// Set up JSX-aware coloring if available
	recolor_range = color_js_es2015_range;
	recolor_from_here = color_c_from_here;

	try_calling("javascript-mode-hook");
	drop_all_colored_regions();
	make_mode();
}

// Enhanced JavaScript mode - default for .js files
javascript_mode()
{
	javascript_es2015_mode();
}

suffix_js()
{
	javascript_mode();
}

// JSX mode for React files
command jsx_mode()
{
	javascript_es2015_mode();
	major_mode = "JSX";
	make_mode();
}

suffix_jsx()
{
	jsx_mode();
}

// TypeScript support
suffix_ts()
{
	javascript_mode();
	compile_buffer_cmd = compile_typescript_cmd;
}

suffix_tsx()
{
	jsx_mode();
	compile_buffer_cmd = compile_typescript_cmd;
}

// JSON mode
command json_mode()
{
	c_mode();
	c_extra_keywords = JS_KEYWORDS;
	major_mode = "JSON";
	strcpy(comment_start, "");
	strcpy(comment_pattern, "");
	recolor_range = color_json_range;
	recolor_from_here = color_c_from_here;
	try_calling("json-mode-hook");
	drop_all_colored_regions();
	make_mode();
}

suffix_json()
{
	json_mode();
}

// Check if text matches an ES2015+ keyword
is_js_es2015_keyword(char *p)
{
	// ES2015+ keywords not in the base JavaScript set
	if (strstr("|let|const|async|await|yield|static|get|set|"
			   "|constructor|debugger|symbol|of|from|as|"
			   "|default|export|", p))
		return 1;
	return 0;
}

// Enhanced JavaScript keyword coloring for ES2015+
int js_keyword_color(int from)
{
	char buf[500];

	if (point - from > sizeof(buf) / sizeof(char) - 10)
		return color_class c_identifier;

	buf[0] = '|';
	grab(from, point, buf + 1);

	// Check for numbers
	if (index("0123456789-.", buf[1]))
		return c_number_color(buf + 1);

	strcpy(buf + point - from + 1, "|");

	// Check standard C/JS keywords
	if (is_c_keyword(buf))
		return color_class c_keyword;

	// Check ES2015+ keywords
	if (is_js_es2015_keyword(buf))
		return color_class c_keyword;

	// Check if it's a function call
	if (color_class c_function != color_class c_identifier && paren_follows())
		return color_class c_function;

	return color_class c_identifier;
}

// Check if we're at the start of a JSX tag
int at_jsx_tag()
{
	save_var point;
	int c = curchar();

	if (c != '<')
		return 0;

	point++;
	c = curchar();

	// Check for closing tag </...>
	if (c == '/')
		point++, c = curchar();

	// Check for fragment <>
	if (c == '>')
		return 1;

	// JSX tag names start with uppercase letter or lowercase (for HTML elements)
	if ((c >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z'))
		return 1;

	return 0;
}

// Color a JSX tag and its contents
void color_jsx_tag(int start)
{
	char tag_name[200];

	point = start + 1;

	// Handle closing tag
	if (curchar() == '/') {
		point++;
		if (parse_string(1, "[A-Za-z][A-Za-z0-9._-]*")) {
			set_character_color(start, point, color_class jsx_tag);
			if (search(1, ">"))
				set_character_color(point - 1, point, color_class jsx_tag);
			return;
		}
	}

	// Handle fragment <>
	if (curchar() == '>') {
		point++;
		set_character_color(start, point, color_class jsx_tag);
		return;
	}

	// Parse opening tag name
	if (!parse_string(1, "[A-Za-z][A-Za-z0-9._-]*", tag_name)) {
		// Not a valid JSX tag - just skip the <
		point = start + 1;
		return;
	}

	int name_end = point;
	set_character_color(start, name_end, color_class jsx_tag);

	// Parse attributes
	while (point < size()) {
		point += parse_string(1, "[ \t\n]*");

		int c = curchar();

		// Self-closing tag
		if (c == '/' && character(point + 1) == '>') {
			point += 2;
			set_character_color(point - 2, point, color_class jsx_tag);
			return;
		}

		// End of opening tag
		if (c == '>') {
			point++;
			set_character_color(point - 1, point, color_class jsx_tag);
			return;
		}

		// Attribute name
		int attr_start = point;
		if (parse_string(1, "[A-Za-z][A-Za-z0-9._-]*")) {
			set_character_color(attr_start, point, color_class jsx_attribute);

			point += parse_string(1, "[ \t\n]*");

			// Attribute value
			if (curchar() == '=') {
				point++;
				point += parse_string(1, "[ \t\n]*");

				int val_start = point;
				if (curchar() == '"') {
					re_search(1, "\"([^\"\\\n]|\\(.|\n))*[\"\n]");
					set_character_color(val_start, point, color_class c_string);
				} else if (curchar() == '\'') {
					re_search(1, "\'([^\'\\\n]|\\(.|\n))*[\'\n]");
					set_character_color(val_start, point, color_class c_string);
				} else if (curchar() == '{') {
					// JSX expression in attribute
					point++;
					int depth = 1;
					int expr_start = point;
					while (depth > 0 && point < size()) {
						if (curchar() == '{')
							depth++;
						else if (curchar() == '}')
							depth--;
						if (depth > 0)
							point++;
					}
					// Color the expression content as JavaScript
					color_narrowed_c_range(expr_start, point);
					if (curchar() == '}')
						point++;
				}
			}
		} else {
			// Couldn't parse attribute - exit and ensure point advances
			break;
		}
	}
	// Ensure point is advanced past start even if we couldn't parse everything
	if (point <= start)
		point = start + 1;
}

// Color a narrowed range using C-mode coloring (for JSX expressions)
// Simplified to avoid recursion issues - colors entire expression as identifier
int color_narrowed_c_range(int start, int end)
{
	if (start < end)
		set_character_color(start, end, color_class c_identifier);
	return end;
}

// Color JavaScript template literals with ${} expressions
// Simplified to avoid recursion - treats entire template as string
void color_js_template_literal(int start)
{
	int p = start;

	if (character(p) != '`')
		return;

	p++; // Skip opening backtick

	// Find the closing backtick
	while (p < size()) {
		int c = character(p);

		if (c == '`') {
			// Found closing backtick
			p++;
			point = p;
			set_character_color(start, p, color_class c_string);
			return;
		} else if (c == '\\' && p + 1 < size()) {
			// Skip escaped character
			p += 2;
		} else {
			p++;
		}
	}

	// Unclosed template literal - color what we have
	point = p;
	set_character_color(start, p, color_class c_string);
}

// Enhanced coloring for ES2015+ JavaScript with JSX support
int color_js_es2015_range(int from, int to)
{
	// Check if we're in a JSX context
	if (c_extra_keywords & JS_KEYWORDS) {
		int start;
		if (get_tagged_region("color-c-as-unit", from, &start) != -1)
			from = start;
	}

	return do_color_js_es2015_range(from, to);
}

int do_color_js_es2015_range(int from, int to)
{
	int t = -1, talk, s, talk_at = 0;
	char pat[300];
	TIMER talk_now;

	trace_enter("do_color_js_es2015_range");
	trace_2i("from,to=", from, to);

	if (from >= to) {
		trace("from >= to, returning");
		return to;
	}

	save_var point, matchstart, matchend;
	c_init_color(from, to);
	point = from;

	trace_2i("point,size=", point, size());

	if (talk = (to - from > 30000))
		time_begin(&talk_now, 50);

	save_var case_fold = 0;

	trace_i("minimal_coloring BEFORE=", minimal_coloring);

	// Enhanced pattern for ES2015+ and JSX
	// Note: < must be escaped as %< outside special contexts
	strcpy(pat, "/<*>|//|^[ \t]*#|[\"'`]|=>|%<[A-Za-z/>]");
	trace_i("minimal_coloring AFTER strcpy=", minimal_coloring);
	if (!minimal_coloring) {
		trace("adding identifier pattern");
		// Note: $ must be escaped as %$ inside character classes
		strcat(pat, "|[A-Za-z_%$][A-Za-z0-9_%$]*"
			   "|-?%.?[0-9]([A-Za-z0-9._']|[Ee]-)*");
	} else {
		trace("SKIPPING identifier pattern - minimal_coloring is ON");
	}

	// Debug: print the actual pattern
	if (TRACE_ENABLED) {
		int saved_buf = bufnum;
		to_buffer("#js-debug#");
		point = size();
		bprintf("PATTERN=[%s]\n", pat);
		bufnum = saved_buf;
	}

	trace("pattern built");

	while (point < to) {
		trace_2i("Loop: point,to=", point, to);
		trace_i("char at point=", character(point));

		if (!re_search(1, pat)) {
			t = size();
			trace("re_search failed");
			trace_i("t=", t);
			break;
		}

		t = matchstart;
		trace_3i("t,matchstart,point=", t, matchstart, point);

		int c = character(point - 1);
		int prev = character(point - 2);
		trace_2i("c,prev=", c, prev);

		// Check for arrow functions =>
		if (c == '>' && prev == '=') {
			// Color both = and > as keyword, but ensure t-1 is valid
			int start = (t > 0) ? t - 1 : t;
			set_character_color(start, point, color_class c_keyword);
			continue;
		}

		// Check for JSX tags
		if (c == '<' || c == '>') {
			point = t;
			if (at_jsx_tag()) {
				color_jsx_tag(t);
				continue;
			}
			point = t + 1;
			continue;
		}

		switch (c) {
		case '/':
			if (prev == '/') {
				// Single-line comment
				while (nl_forward() && character(point - 2) == '\\')
					;
				// Color both slashes, but ensure t-1 is valid
				int start = (t > 0) ? t - 1 : t;
				set_character_color(start, point, color_class c_comment);
			} else if (curchar() == '*') {
				// Multi-line comment
				point++;
				search(1, "*/");
				set_character_color(t, point, color_class c_comment);
				if (get_character_color(point, (int *) 0, &s)
					  == color_class c_comment && s > to) {
					c_init_color(point, s);
					to = s;
				}
			} else {
				// Could be regex literal - simplified detection
				// Full regex detection would need more context analysis
				point = t + 1;
			}
			break;

		case '#':
			// Preprocessor (not common in JS but supported)
			c_preproc_color();
			set_character_color(t, point, color_class c_preprocessor);
			break;

		case '"':
			point = t;
			re_search(1, "\"([^\"\\\n]|\\(.|\n))*[\"\n]");
			set_character_color(t, point, color_class c_string);
			if (get_character_color(point, (int *) 0, &s) ==
				color_class c_string && s > to)
				c_init_color(point, to = s);
			break;

		case '\'':
			point = t;
			re_search(1, "\'([^\'\\\n]|\\(.|\n))*[\'\n]");
			set_character_color(t, point, color_class c_string);
			break;

		case '`':
			// Template literal (already supported by base C mode)
			color_js_template_literal(t);
			break;

		default:
			// Identifier, keyword, or number
			trace_2i("default: t,point=", t, point);
			set_character_color(t, point, js_keyword_color(t));
			trace_i("after color, point=", point);
			break;
		}

		trace_i("loop end, point=", point);

		if (talk && point > talk_at + 30000 && time_done(&talk_now)) {
			note("Coloring JavaScript: %d%% complete...",
				 muldiv(point - from, 100, to - from));
			talk_at = point;
		}
	}

	// Ensure we never call c_init_color with start > end
	if (point < to)
		point = to;
	trace_2i("c_init_color: to,point=", to, point);
	c_init_color(to, point);
	if (talk_at)
		note("");
	trace_i("returning=", point);
	return point;
}

// Simplified JSON coloring (stricter than JavaScript)
int color_json_range(int from, int to)
{
	int t = -1;
	char pat[200];

	if (from >= to)
		return to;

	save_var point, matchstart, matchend;
	c_init_color(from, to);
	point = from;
	save_var case_fold = 0;

	strcpy(pat, "[\"']");
	if (!minimal_coloring)
		strcat(pat, "|[A-Za-z_][A-Za-z0-9_]*|-?%.?[0-9]([0-9._]|[Ee]-)*");

	while (point < to) {
		if (!re_search(1, pat)) {
			t = size();
			break;
		}

		t = matchstart;
		switch (character(point - 1)) {
		case '"':
			point = t;
			re_search(1, "\"([^\"\\\n]|\\(.|\n))*[\"\n]");
			set_character_color(t, point, color_class c_string);
			break;

		case '\'':
			point = t;
			re_search(1, "\'([^\'\\\n]|\\(.|\n))*[\'\n]");
			set_character_color(t, point, color_class c_string);
			break;

		default:
			// In JSON, only true/false/null are keywords
			char buf[100];
			buf[0] = '|';
			grab(t, point, buf + 1);
			if (index("0123456789-.", buf[1]))
				set_character_color(t, point, c_number_color(buf + 1));
			else {
				strcpy(buf + point - t + 1, "|");
				if (strstr("|true|false|null|", buf))
					set_character_color(t, point, color_class c_keyword);
				else
					set_character_color(t, point, color_class c_identifier);
			}
			break;
		}
	}

	c_init_color(to, t);
	return point;
}

// Hook for mode customization
javascript_mode_hook()
{
	// User can override this in their config
}

json_mode_hook()
{
	// User can override this in their config
}
