// Test file for ES2015+ JavaScript features
// Open this in Epsilon to verify syntax highlighting

// ============================================
// ES2015+ Variable Declarations
// ============================================

let mutableVariable = 42;
const immutableConstant = 'Hello';
var oldStyleVar = true;

// ============================================
// Arrow Functions
// ============================================

const simpleArrow = () => console.log('Simple');
const arrowWithParam = (x) => x * 2;
const arrowWithBlock = (a, b) => {
    const sum = a + b;
    return sum;
};

const multiLine = (x, y) =>
    x + y;

// ============================================
// Template Literals
// ============================================

const name = 'World';
const greeting = `Hello, ${name}!`;
const multiLineTemplate = `
    This is a
    multi-line
    template literal
`;

const nested = `Outer ${`inner ${42}`} text`;
const expression = `Result: ${1 + 2 + 3}`;

// ============================================
// Classes
// ============================================

class Animal {
    constructor(name) {
        this.name = name;
    }

    speak() {
        console.log(`${this.name} makes a sound`);
    }

    static create(name) {
        return new Animal(name);
    }

    get fullName() {
        return this.name;
    }

    set fullName(value) {
        this.name = value;
    }
}

class Dog extends Animal {
    constructor(name, breed) {
        super(name);
        this.breed = breed;
    }

    speak() {
        console.log(`${this.name} barks`);
    }
}

// ============================================
// Async/Await
// ============================================

async function fetchData(url) {
    try {
        const response = await fetch(url);
        const data = await response.json();
        return data;
    } catch (error) {
        console.error('Error:', error);
    }
}

const asyncArrow = async () => {
    await someAsyncOperation();
    return 'done';
};

// ============================================
// Generators
// ============================================

function* generateSequence() {
    yield 1;
    yield 2;
    yield 3;
}

function* infiniteGenerator() {
    let i = 0;
    while (true) {
        yield i++;
    }
}

// ============================================
// Destructuring
// ============================================

// Array destructuring
const [first, second, ...rest] = [1, 2, 3, 4, 5];

// Object destructuring
const { name: userName, age } = { name: 'Alice', age: 30 };

// Function parameter destructuring
function processUser({ name, email }) {
    console.log(name, email);
}

// ============================================
// Spread and Rest Operators
// ============================================

const arr1 = [1, 2, 3];
const arr2 = [...arr1, 4, 5, 6];

const obj1 = { a: 1, b: 2 };
const obj2 = { ...obj1, c: 3 };

function sum(...numbers) {
    return numbers.reduce((a, b) => a + b, 0);
}

// ============================================
// For-of Loops
// ============================================

for (const item of [1, 2, 3]) {
    console.log(item);
}

for (const char of 'hello') {
    console.log(char);
}

// ============================================
// Modules (ES6)
// ============================================

// Import statements
import React from 'react';
import { useState, useEffect } from 'react';
import * as Utils from './utils';
import DefaultExport, { namedExport } from './module';

// Export statements
export const PI = 3.14159;
export function calculate(x) {
    return x * 2;
}

export default class MyClass {
    constructor() {
        this.value = 0;
    }
}

export { calculate as calc };
export { default as MyDefault } from './other';

// ============================================
// Enhanced Object Literals
// ============================================

const x = 10, y = 20;
const point = {
    x,  // shorthand property
    y,
    // Method shorthand
    move() {
        console.log('Moving');
    },
    // Computed property names
    ['prop_' + 42]: true,
    // Getter and setter
    get z() {
        return this.x + this.y;
    },
    set z(value) {
        this.x = value / 2;
        this.y = value / 2;
    }
};

// ============================================
// Symbols
// ============================================

const sym1 = Symbol();
const sym2 = Symbol('description');
const sym3 = Symbol.for('shared');

const obj = {
    [sym1]: 'value',
    [Symbol.iterator]: function* () {
        yield 1;
        yield 2;
    }
};

// ============================================
// Default Parameters
// ============================================

function greet(name = 'Guest', greeting = 'Hello') {
    return `${greeting}, ${name}!`;
}

const multiply = (a, b = 1) => a * b;

// ============================================
// Promises
// ============================================

const promise = new Promise((resolve, reject) => {
    setTimeout(() => {
        resolve('Success');
    }, 1000);
});

promise
    .then(result => console.log(result))
    .catch(error => console.error(error))
    .finally(() => console.log('Done'));

// ============================================
// Map and Set
// ============================================

const map = new Map();
map.set('key1', 'value1');
map.set('key2', 'value2');

for (const [key, value] of map) {
    console.log(key, value);
}

const set = new Set([1, 2, 3, 4, 4, 5]);
set.add(6);

// ============================================
// Optional Chaining and Nullish Coalescing
// (ES2020, but commonly used)
// ============================================

const user = {
    name: 'Alice',
    address: {
        street: 'Main St'
    }
};

const street = user?.address?.street;
const zip = user?.address?.zip ?? 'Unknown';

// ============================================
// Debugging
// ============================================

debugger;  // Breakpoint keyword

// ============================================
// Comments
// ============================================

// Single-line comment

/*
 * Multi-line comment
 * with multiple lines
 */

/* Inline comment */

// ============================================
// Mixed Features
// ============================================

const complexFunction = async ({
    name = 'default',
    ...options
}) => {
    debugger;
    const { value } = await fetchData(name);
    const result = `Result: ${value}`;

    for (const option of Object.keys(options)) {
        console.log(option);
    }

    return result;
};

export { complexFunction as default };
