// Test file for JSX/React features
// Open this in Epsilon to verify JSX syntax highlighting

import React, { useState, useEffect } from 'react';
import PropTypes from 'prop-types';

// ============================================
// Simple JSX Elements
// ============================================

// Self-closing tag
const SimpleComponent = () => <div />;

// With children
const WithChildren = () => <div>Hello World</div>;

// Nested elements
const Nested = () => (
    <div>
        <h1>Title</h1>
        <p>Paragraph</p>
    </div>
);

// ============================================
// JSX Attributes
// ============================================

const WithAttributes = () => (
    <div
        className="container"
        id="main"
        data-testid="test"
        style={{ color: 'red', fontSize: '14px' }}
    >
        Content
    </div>
);

const WithEventHandlers = () => (
    <button
        onClick={() => console.log('Clicked')}
        onMouseEnter={handleMouseEnter}
        disabled={false}
    >
        Click Me
    </button>
);

// ============================================
// JSX Expressions
// ============================================

const WithExpressions = ({ count }) => (
    <div>
        <p>Count: {count}</p>
        <p>Double: {count * 2}</p>
        <p>Message: {count > 5 ? 'High' : 'Low'}</p>
    </div>
);

// ============================================
// Conditional Rendering
// ============================================

const ConditionalRendering = ({ isLoggedIn, user }) => (
    <div>
        {isLoggedIn ? (
            <p>Welcome, {user.name}!</p>
        ) : (
            <button>Login</button>
        )}

        {isLoggedIn && <Dashboard />}

        {!isLoggedIn && <Login />}
    </div>
);

// ============================================
// Lists and Keys
// ============================================

const ListComponent = ({ items }) => (
    <ul>
        {items.map((item) => (
            <li key={item.id}>
                {item.name}
            </li>
        ))}
    </ul>
);

const NestedList = ({ data }) => (
    <div>
        {data.map((section) => (
            <div key={section.id}>
                <h2>{section.title}</h2>
                <ul>
                    {section.items.map((item) => (
                        <li key={item.id}>{item.text}</li>
                    ))}
                </ul>
            </div>
        ))}
    </div>
);

// ============================================
// JSX Fragments
// ============================================

const WithFragment = () => (
    <>
        <h1>Title</h1>
        <p>Description</p>
    </>
);

const LongFormFragment = () => (
    <React.Fragment>
        <dt>Term</dt>
        <dd>Definition</dd>
    </React.Fragment>
);

// ============================================
// Component Composition
// ============================================

const Header = ({ title }) => (
    <header>
        <h1>{title}</h1>
        <nav>
            <a href="/">Home</a>
            <a href="/about">About</a>
        </nav>
    </header>
);

const Footer = () => (
    <footer>
        <p>&copy; 2025 My Company</p>
    </footer>
);

const Layout = ({ children }) => (
    <div className="layout">
        <Header title="My App" />
        <main>{children}</main>
        <Footer />
    </div>
);

// ============================================
// Hooks in JSX
// ============================================

const Counter = () => {
    const [count, setCount] = useState(0);

    useEffect(() => {
        document.title = `Count: ${count}`;
    }, [count]);

    return (
        <div>
            <p>Count: {count}</p>
            <button onClick={() => setCount(count + 1)}>
                Increment
            </button>
            <button onClick={() => setCount(count - 1)}>
                Decrement
            </button>
        </div>
    );
};

// ============================================
// Complex Nested JSX
// ============================================

const ComplexComponent = ({ data, onUpdate }) => {
    const [selected, setSelected] = useState(null);
    const [loading, setLoading] = useState(false);

    const handleClick = async (item) => {
        setLoading(true);
        try {
            await onUpdate(item);
            setSelected(item);
        } catch (error) {
            console.error(error);
        } finally {
            setLoading(false);
        }
    };

    return (
        <div className="complex-component">
            <header>
                <h1>Complex Component</h1>
                {loading && <Spinner />}
            </header>

            <div className="content">
                {data.map((item) => (
                    <Card
                        key={item.id}
                        item={item}
                        isSelected={selected?.id === item.id}
                        onClick={() => handleClick(item)}
                    >
                        <CardHeader title={item.title} />
                        <CardBody>
                            <p>{item.description}</p>
                            {item.tags.map((tag) => (
                                <Tag key={tag} name={tag} />
                            ))}
                        </CardBody>
                        <CardFooter>
                            <button>Edit</button>
                            <button>Delete</button>
                        </CardFooter>
                    </Card>
                ))}
            </div>

            {selected && (
                <Modal onClose={() => setSelected(null)}>
                    <h2>{selected.title}</h2>
                    <p>{selected.description}</p>
                </Modal>
            )}
        </div>
    );
};

// ============================================
// JSX with Spread Attributes
// ============================================

const Button = (props) => (
    <button {...props} className={`btn ${props.className || ''}`}>
        {props.children}
    </button>
);

const UsesButton = () => (
    <div>
        <Button type="submit" onClick={handleSubmit}>
            Submit
        </Button>
    </div>
);

// ============================================
// JSX Comments
// ============================================

const WithComments = () => (
    <div>
        {/* This is a JSX comment */}
        <p>Content</p>

        {/*
            Multi-line
            JSX comment
        */}
        <p>More content</p>
    </div>
);

// ============================================
// Custom Components (PascalCase)
// ============================================

const UserCard = ({ user }) => (
    <div className="user-card">
        <UserAvatar src={user.avatar} alt={user.name} />
        <UserInfo name={user.name} email={user.email} />
        <UserActions userId={user.id} />
    </div>
);

// ============================================
// HTML Elements (lowercase)
// ============================================

const HtmlElements = () => (
    <div>
        <p>Paragraph</p>
        <span>Span</span>
        <a href="#">Link</a>
        <img src="image.jpg" alt="Description" />
        <input type="text" value="Input" />
        <select>
            <option value="1">Option 1</option>
            <option value="2">Option 2</option>
        </select>
    </div>
);

// ============================================
// Props and PropTypes
// ============================================

const Greeting = ({ name, age, isActive }) => (
    <div className={isActive ? 'active' : 'inactive'}>
        <h1>Hello, {name}!</h1>
        <p>Age: {age}</p>
    </div>
);

Greeting.propTypes = {
    name: PropTypes.string.isRequired,
    age: PropTypes.number,
    isActive: PropTypes.bool
};

Greeting.defaultProps = {
    age: 0,
    isActive: false
};

// ============================================
// JSX in Expressions
// ============================================

const jsxInArray = [
    <div key="1">First</div>,
    <div key="2">Second</div>,
    <div key="3">Third</div>
];

const jsxInObject = {
    header: <Header />,
    body: <Body />,
    footer: <Footer />
};

// ============================================
// Complex Template Literals in JSX
// ============================================

const TemplateInJSX = ({ name, role }) => (
    <div>
        <p className={`user-${role.toLowerCase()}`}>
            {`Welcome, ${name}!`}
        </p>
        <p>
            {`You are logged in as ${role}`}
        </p>
    </div>
);

// ============================================
// Export Default Component
// ============================================

export default function App() {
    return (
        <Layout>
            <Counter />
            <ComplexComponent
                data={[]}
                onUpdate={async (item) => {
                    console.log('Updating', item);
                }}
            />
        </Layout>
    );
}

// Named exports
export { Counter, ComplexComponent, Greeting };
