// fill.e
//
// Fri, 06/12/1992 15:07:19
//
// Purpose:
//
//		One of the most tedious of the number of standard things done
//		to code is writing comments.  It is one of the most important
//		steps to achieve a maintainable and solid piece of code
//		(aside from the obvious writing of the code itself).  Most
//		tools are not optimized for this process at all since the
//		usual method for embedding comments and documentation within
//		code is via a delimiter at the beginning of a line such as a
//		';' or '#' or " " etc.  This comment prefix is a royal pain
//		in the neck to deal with in terms of formatting and
//		reformatting text.  This Epsilon V6.0 extension is designed
//		to address this issue.
//
//		This module hooks into the existing fill mode in Epsilon and
//		extends it to allow the dynamic creation of the fill_prefix
//		currently in use and then allows the standard use of the
//		commands fill_region, fill_paragraph, forward_paragraph and
//		backward_paragraph with this fill prefix.
//
//		The fill prefix may contain whitespace before and after the
//		prefix characters (which may be nothing at all).  This
//		definition allows the specification of a standard indent
//		which simulates the presence of a left margin and allows
//		formatting and reformatting within this new margin.
//
// Usage:
//
//		To obtain the most convenient usage from this module I would
//		suggest binding the fill_mode command to a readily available
//		key (I use F3).  This allows simple and fast entrance to fill
//		mode.  The intended usage is starting the comment or prefix
//		and at the end of the first line, enter fill mode.  At this
//		point, the fill_prefix is computed from the current line and
//		this fill prefix is used for all fill mode commands while in
//		fill mode.
//
//		When the comment, paragraph or whatever is finished, the fast
//		keystroke removes the computed fill_prefix and the commands
//		revert to normal behavior.  This approach allows varying
//		indent levels with very little effort and removes the
//		distraction of having to deal with the return, indent and
//		prefix write that many of us are all too familiar with (even
//		if the prefix is just a tab).
//
//		There is no special setup for this extension, simply load it
//		in, save the state file using write_state, and you are up and
//		running.
//
// Details:
//
//		Although I see few reasons for it, to allow the user to
//		enable or disable the auto prefix computation there is a
//		global variable called auto_fill_prefix which is set to TRUE
//		(1) to compute the fill_prefix on entry into fill mode and is
//		set to FALSE (0) to disable auto prefix grab...
//
//		If auto_fill_prefix is FALSE then all routines defined here
//		whether in fill mode or not will use the fill_prefix that you
//		have permanently defined.  I claim this to be desirable
//		behavior assuming you don't want the prefix computed for you.
//
//		if a auto fill_prefix is being used then the tex directives
//		are not ignored during the fill.  Normal behavior is to leave
//		them alone.
//
//		This implementation uses is_word_char() to determine the fill
//		prefix.  The implications of this is that no word characters
//		may be used within the fill_prefix (at least when
//		auto_fill_prefix is TRUE.  In order to allow the use of a
//		current word character, you will need to redefine the
//		word-pattern variable for the buffer in question which
//		determines this information.
//
//		This code has been written so that there are no problem with
//		prefixes which have been trimmed on the right in the case
//		where the prefix has whitespace on the right (most cases are
//		of this type ie. " * ").  This is accomplished by performing
//		searches based only on the fill prefix after all trailing
//		whitespace has been deleted.  This should not create any
//		unexpected (or rather, unwanted behavior).  A side effect of
//		this is that a whitespace prefix will look the same as no
//		prefix for searching for regions and will behave the same as
//		if fill mode were not enabled.
//
//		If a fill prefix is defined and you fill a region or
//		paragraph of non prefixed text then the text will be
//		realigned and filled.  This means that you can indent and
//		refill paragraphs from the left hand margin easily or take a
//		large block of docs and embed it in the code very easily as a
//		comment.
//
//		The trimming of the fill prefix for searches also allows
//		correction of sections where for some reason, the text after
//		the fill_prefix has been modified by the insertion of or
//		deletion of whitespace.  This, as a side affect, means that
//		paragraphs may be (un)indented additional amounts or
//		reformatted and gives a simple way to obtain a left margin
//		(simply use a fill prefix of a tab or 2 or 3 ...
//
//		This extension will honor the paragraph definition convention
//		determined by indents_seperate_paragraphs.  The code as
//		currently written however will poorly handle mixed tabs and
//		spaces determining paragraph beginnings based on indent if
//		indents_separate_paragraphs is true (tabs are a pain is the
//		neck to compute a string length for).
//
//		Lugaru code has a fence post bug in format.e , hard to find
//		in actuality.  If the current start of paragraph is at the
//		beginning of the buffer with one line before it then the
//		cursor winds up in the wrong spot and paragraph fill for that
//		paragraph will result in deletion of the first new line
//		character.  This bug would require rewrite of
//		backward_paragraph().  If you are using the auto fill prefix
//		code and there is a non zero length fill_prefix defined then
//		you will not see this bug.  Since a zero length fill_prefix
//		will use the original Lugaru code for backward_paragraph you
//		may see it in this case...
//
// Some future directions:
//
//		1) It may make sense to alter the indent_region and
//		indent_rigidly commands to correctly ignore the fill_prefix.
//
//		2) *Correctly* handling added whitespace at the paragraph
//		beginning should not be all that difficult but is beyond the
//		scope I wish to deal with here.
//
//
// The modifications to this code are dedicated to the public domain
// with the caveat that Lugaru is welcome to use this within their
// distribution source code which is supplied with Epsilon.  Use it
// any way you want for whatever purposes.  Portions are Copyright
// Lugaru Software...
//
//
//	John Kercheval
//	15563 10th Ave NE
//	Seattle, WA 98155-6209
//	INTERNET: johnk@wrq.com
//	Compu$erve: 72450,3702
//
//
// Modification History:
//
//		Fri, 06/12/1992 15:07:19 V1.0 initial release.
//
//		Thu, 06/25/1992 13:17:06 V1.1 fixed bug in to_region_border
//		when dealing with a region at the end of the buffer.
//
//		Tue, 07/07/1992 23:52:31 V1.2 fixed bug if dealing with large
//		words within a paragraph if the fill_prefix had leading
//		whitepace and a fill was occuring on a paragraph of fill
//		without the prefix.  A counting error which was quite deadly
//		in fill_region.  Reformat code using BSD indent.
//
//		Mon, 07/27/1992  21:20:41 V1.3 save and restore the buffer
//		specific fill_prefixes via the session save and restore
//		hooks.
//
//		Sat, 02/06/1993 10:50:55 V1.4 update for use with
//		V6.5 of Epsilon.
//
//		Thu, 07/08/1993 10:57:00 V1.5 Aaron A. Collins - CIS
//		70324,3200 - auto-fill-mode hangs if run in a brand-new empty
//		buffer.  Made it bail out if it is found that the point is
//		not moving anymore when advanced.
//		
//


//
// some amount of Epsilon original code so...
//
/************************************************************************
* "Epsilon" is a registered trademark licensed to Lugaru Software, Ltd. *
*   "EEL" and "Lugaru" are trademarks of Lugaru Software, Ltd.          *
*                                                                       *
*  Copyright (C) 1985, 1992 Lugaru Software Ltd.  All rights reserved.  *
*                                                                       *
* Limited permission is hereby granted to reproduce and modify this     *
* copyrighted material provided that the resulting code is used only in *
* conjunction with Lugaru products and that this notice is retained in  *
* any such reproduction or modification.                                *
************************************************************************/


#include "eel.h"

#ifndef BOOL
#define BOOL int
#define TRUE 1
#define FALSE 0
#endif

#define FILL_PREFIX_MAX_LENGTH 256


////////////////////////////////////////////////////////////////////////////
//
// Globals which modify behavior of the filling code.
//

user BOOL auto_fill_prefix = TRUE;	// true if prefix auto determined

user buffer char fill_prefix[FILL_PREFIX_MAX_LENGTH] = "";


////////////////////////////////////////////////////////////////////////////
//
// auto_fill_mode() uses the current line to determine the fill
// prefix.  This behavior may be altered by setting the variable
// auto_fill_prefix to 0.
//

/* make space break line or not */

command
auto_fill_mode()
{
	BOOL done = FALSE;
	char tmpstr[FILL_PREFIX_MAX_LENGTH + 20];
	int previousPoint;
	
	save_spot point,
	    mark;

	fill_mode = (has_arg ? (iter != 0) : !fill_mode);
	if (auto_fill_prefix) {
		fill_prefix[0] = '\0';
		if (fill_mode) {
			to_begin_line();
			mark = point;
			while (!done) {
				previousPoint = point;
				if (!is_word_char(point) && curchar() != '\n') {
					point++;
					if (point == previousPoint) {
						//
						// we have reached the end of the buffer, we
						// are done.
						//
						done = TRUE;
					}
				} else {
					done = TRUE;
				}
			}
			grab(mark, point, fill_prefix);
			sprintf(tmpstr, "Fill Prefix:  \"%s\"", fill_prefix);
			say(tmpstr);
		}
	} else {
		say("");
	}

	make_mode();
	iter = 1;
}


////////////////////////////////////////////////////////////////////////////
//
// Move back across the fill_prefix and move over until we know where
// we need to break the line.
//

/* determine where to break current line given right margin */
pick_break(col)
	int col;
{
	int orig = point,
	    start,
	    colpos;

	to_begin_line();
	point += strlen(fill_prefix);
	re_search(1, "[ \t]*");		/* skip indentation */
	start = point;
	move_to_column(col);		/* find first space before col */
	colpos = point;
	if (re_search(-1, "[ \t]") && point > start) {
		point++;
		return 1;
	}
	point = colpos;				/* else find first space after col */
	if (re_search(1, "[ \t]") && point < orig) {
		point--;
		return 1;
	}
	point = orig;
	return 0;
}


////////////////////////////////////////////////////////////////////////////
//
// Break the line and add the fill prefix at the beginning of the
// next line.
//

break_line_here()
{								/* replace whitespace with \n and indent */
	delete_horizontal_space();
	insert('\n');
	stuff(fill_prefix);
	if (auto_fill_indents && indenter && !fill_prefix[0])
		(*indenter) ();
	if (display_column > 0)
		display_column = 0;
}


////////////////////////////////////////////////////////////////////////////
//
// skip_prefix_whitespace() will move over empty lines and
//		whitespace which contain only a prefix or nothing.  The
//		direction will be determined by the parameter dir according
//		to the standard direction parameters for search, re_search,
//		etc.  The point will be set at the beginning of the first non
//		whitespace line.
//

skip_prefix_whitespace(dir)
	int dir;
{
	char tmpstr[FILL_PREFIX_MAX_LENGTH];
	BOOL found;
	int stripped_prefix_length,
	    this_point;

	//
	// setup clipped prefix for trimmed prefixed regions
	//
	strcpy(tmpstr, fill_prefix);
	stripped_prefix_length = strlen(fill_prefix);
	while (stripped_prefix_length &&
		   ((tmpstr[stripped_prefix_length - 1] == ' ') ||
			(tmpstr[stripped_prefix_length - 1] == '\t'))) {
		stripped_prefix_length--;
	}

	//
	// move over prefixes and whitespace
	//
	if (current_column() == (dir < 0 ? stripped_prefix_length : 0)) {
		grab(point, point + (dir < 0 ? -stripped_prefix_length :
							 stripped_prefix_length), tmpstr);
		if (!strncmp(tmpstr, fill_prefix, stripped_prefix_length))
			point += (dir < 0 ? -stripped_prefix_length :
					  stripped_prefix_length);
	}
	found = FALSE;
	while (!found) {
		this_point = point;
		if (re_search((dir < 0 ? -1 : 1), "[ \t\n]*")) {
			if (matchstart == this_point) {
				grab(point, point + (dir < 0 ? -stripped_prefix_length :
									 stripped_prefix_length), tmpstr);
				if (!strncmp(tmpstr, fill_prefix, stripped_prefix_length))
					point += (dir < 0 ? -stripped_prefix_length :
							  stripped_prefix_length);
				else
					found = TRUE;
			} else
				found = TRUE;
		} else {
			found = TRUE;
		}
		if (this_point == point)
			found = TRUE;
	}
}


////////////////////////////////////////////////////////////////////////////
//
//	to_region_border() will move to the beginning or end of a paragraph
//		which may be delineated by fill_prefix characters.  The
//		function will return TRUE if the beginning of the region is
//		at the edge (beginning or end) of the buffer.
//

BOOL
to_region_border(dir)
	int dir;
{
	char tmpstr[FILL_PREFIX_MAX_LENGTH];
	BOOL found = FALSE,
	    border = FALSE;
	int stripped_prefix_length,
	    fill_prefix_length;
	int hold_point,
	    previous_hold_point,
	    this_point;

	//
	// setup clipped prefix for trimmed prefixed regions
	//
	strcpy(tmpstr, fill_prefix);
	fill_prefix_length = stripped_prefix_length = strlen(fill_prefix);
	while (stripped_prefix_length &&
		   ((tmpstr[stripped_prefix_length - 1] == ' ') ||
			(tmpstr[stripped_prefix_length - 1] == '\t'))) {
		stripped_prefix_length--;
	}

	//
	// move over any white space
	//
	skip_prefix_whitespace(dir < 0 ? -1 : 1);

	//
	// Crawl in the direction of travel until we have an indent or an
	// empty line.
	//
	hold_point = point;
	while (!found && !border) {
		if (dir < 0) {
			if (!nl_reverse()) {
				point = 0;
				border = TRUE;
			}
		} else {
			if (!nl_forward()) {
				point = size();
				border = TRUE;
			}
		}
		previous_hold_point = hold_point;
		hold_point = point;
		if (dir < 0 && !border)
			point++;
		grab(point, point + stripped_prefix_length, tmpstr);
		if (!strncmp(tmpstr, fill_prefix, stripped_prefix_length))
			point += stripped_prefix_length;
		if (curchar() == '\n' || point == size()) {
			found = TRUE;
			if (dir < 0)
				hold_point++;
		} else {
			this_point = point;
			if (re_search(1, "[ \t]*")) {
				if (curchar() == '\n') {
					if (dir < 0)
						hold_point = previous_hold_point;
					found = TRUE;
				}
				if (matchstart == this_point) {
					if (indents_separate_paragraphs &&
						((matchend - matchstart) >
						 (fill_prefix_length -
						  stripped_prefix_length))) {
						found = TRUE;
					}
				}
			}
		}
		point = hold_point;
		if (found && dir < 0) {
			nl_forward();
		}
	}
	return border;
}


////////////////////////////////////////////////////////////////////////////
//
// forward_paragraph and backward_paragraph both are be modified to
// be aware of the prefix determined during mode start.  The
// paragraph begin is determined by the whitespace found AFTER the
// original fill_prefix on the line.
//

command
forward_paragraph()
	on  reg_tab[ALT(']')], reg_tab[NUMALT(KEYDOWN)]
{
	if (!fill_prefix[0]) {
		re_search(1, ".");
		if (tex_paragraphs) {	/* skip formatter lines */
			to_begin_line();
			re_search(1, "([@.\\].*\n)*");
			re_search(1, "[^ \t\n@.\\]");
		}
		if (re_search(1, indents_separate_paragraphs ? "\n[ \t\n]|\f"
					  : "(\n[ \t]*\n|\f)")) {
			point--;
			if (tex_paragraphs && curchar() != '\f')
				re_search(-1, "^([@.\\].*\n)*");
		} else
			point = size();
	} else {
		to_region_border(1);
	}
}

command
backward_paragraph()
	on  reg_tab[ALT('[')], reg_tab[NUMALT(KEYUP)]
{
	if (!fill_prefix[0]) {
		if (tex_paragraphs && character(point - 1) == '\n')
			re_search(-1, "^([@.\\].*\n)*");
		re_search(-1, "[^ \t\n]");
		if (indents_separate_paragraphs) {
			if (re_search(-1, "\n[ \t\n]|\f"))
				re_search(1, "\n*");
		} else
			re_search(-1, "(\n[ \t]*\n|\f)!");
		if (tex_paragraphs && curchar() != '\f')
			re_search(1, "^([@.\\\f].*\n)*");
	} else {
		to_region_border(-1);
	}
}


////////////////////////////////////////////////////////////////////////////
//
// fill_paragraph has a minor modification to prevent it from
// dealing with leading whitespace in a paragraph as a special case.
//

command
fill_paragraph()
	on  reg_tab[ALT('q')]
{
	int end,
	    start = point;

	iter = 0;
	point--;
	forward_paragraph();
	end = point - 1;
	backward_paragraph();
	if (point < end)
		region_fill(point, end);
	if (start > size())
		start = size();
	point = start;
	if (display_column > 0)
		display_column = 0;
	fix_window_start();			/* win shouldn't start in middle of line */
}


////////////////////////////////////////////////////////////////////////////
//
// move from paragraph to paragraph and fill each block seperately
//

command
fill_region()
{
	int orig = point,
	    start,
	    atend = 0;
	spot end = alloc_spot();

	//
	// order point and mark
	//
	if (point > (*end = mark))
		*end = point, orig = point = mark;

	//
	// march through the region paragraph by paragraph
	//
	while (point < *end && !atend) {
		start = point;
		atend = to_region_border(1);
		if (point > *end + 1)
			point = *end + 1;
		region_fill(start, point - 1);
		skip_prefix_whitespace(1);
		to_begin_line();
	}
	mark = orig;
	point = *end;
	free_spot(end);
	if (display_column > 0)
		display_column = 0;
	fix_window_start();
}


////////////////////////////////////////////////////////////////////////////
//
// Strip all tab and extraneous whitespace and fill_prefixes.
// Starting from the beginning of the region add fill_prefixes (if at
// the beginning of the line) and march through the block.
//

region_fill(a, b)
	int a,
	    b;
{
	int start,
	    atend,
	    len,
	    startcol;
	int fill_prefix_length,
	    stripped_prefix_length;
	char tmpstr[FILL_PREFIX_MAX_LENGTH];

	if (a > b)					/* switch so a < b */
		start = a, a = b, b = start;
	save_spot point = a;

	startcol = current_column();
	save_var narrow_start = a,
	    narrow_end = size() - b;
	save_var abort_searching = 0;

	//
	// setup clipped prefix for trimmed prefixed regions
	//
	strcpy(tmpstr, fill_prefix);
	fill_prefix_length = stripped_prefix_length = strlen(fill_prefix);
	while (stripped_prefix_length &&
		   ((tmpstr[stripped_prefix_length - 1] == ' ') ||
			(tmpstr[stripped_prefix_length - 1] == '\t'))) {
		stripped_prefix_length--;
	}

	point = 0;					/* remove previous line breaks and fill_prefixes */

	//
	// strip initial prefix if present and if at first column
	//
	if (startcol == 0) {
		grab(point, point + fill_prefix_length, tmpstr);
		if (!strcmp(tmpstr, fill_prefix)) {
			delete(point, point + fill_prefix_length);
		} else if (!strncmp(tmpstr, fill_prefix, stripped_prefix_length)) {
			delete(point, point + stripped_prefix_length);
		}
	}
	//
	// Move over initial paragraph indentation
	//
	while (curchar() == ' ' || curchar() == '\t') {
		point++;
	}

	//
	// strip prefixes and newlines, formalize whitespace between words
	//
	while (re_search(1, "([ \t][ \t\n]*)|(\n)")) {
		len = point - matchstart;
		atend = character(matchend - 1) == '\n';
		delete(matchstart, point);
		if (atend) {
			grab(point, point + fill_prefix_length, tmpstr);
			if (!strcmp(tmpstr, fill_prefix)) {
				delete(point, point + fill_prefix_length);
			} else if (!strncmp(tmpstr, fill_prefix, stripped_prefix_length)) {
				delete(point, point + stripped_prefix_length);
			}
		}
		while (curchar() == ' ' || curchar() == '\t') {
			delete(point, point + 1);
			len++;
		}
		if ((len >= 2 || atend) && parse_string(-1, "[.?!][])'\"]*",
												NULL))
			stuff("  ");
		else
			insert(' ');
	}

	//
	// place newlines and prefixes back into buffer
	//
	point = 0;
	for (;;) {
		if (startcol == 0)		// place the fill_prefix if at
			stuff(fill_prefix);	// beginning of line
		start = point;
		move_to_column(margin_right - startcol);
		if (point >= size() - narrow_end)
			break;
		grab(start, point, tmpstr);
		if (strstr(tmpstr, " "))
			re_search(-1, " +");
		if (!re_search(1, " +"))
			break;
		delete(matchstart, point);
		insert('\n');
		startcol = 0;
	}
	build_first = 1;			/* redisplay hint */
}


////////////////////////////////////////////////////////////////////////////
//
// make_session_hook and restore_session_hook save the current
// fill variables to the session file when it is written or read
// and then call any other defined session hooks.
//

#define FILL_START_DELIM	"<<SES!@!FILL_START>>"
#define FILL_END_DELIM		"<<SES!@!FILL_END>>"

new_fill_make_session_hook(b)
{
	int i;

	save_var bufnum = b;

	point = size();

	buf_printf(b, "\n%s", FILL_START_DELIM);
	for (i = buf_list(0, 0); i; i = buf_list(1, 1)) {
		bufnum = i;
		if (i != b && *bufname != '-' && !is_dired_buf() && size()
			&& *filename && strcmp(bufname, PROCBUF)) {
			buf_printf(b, "\n%s\n%s",
					   filename, fill_prefix);
		}
	}
	buf_printf(b, "\n%s\n%s\n", FILL_END_DELIM, FILL_END_DELIM);

	//
	// call the previous hook
	//
	fill_make_session_hook(b);
}

REPLACE_FUNC("fill", "make_session_hook")

new_fill_restore_session_hook(b)
{
	char fname[FNAMELEN];
	char prefix[FILL_PREFIX_MAX_LENGTH];
	int old_point = point;

	save_var bufnum = b;

	//
	// obtain the fill prefixes
	//
	point = 0;
	if (!search(1, FILL_START_DELIM)) {
		point = old_point;
	} else {
		nl_forward();
		do {
			grab_line(b, fname);
			grab_line(b, prefix);
			if (look_file(fname)) {
				strcpy(fill_prefix, prefix);
			}
		} while (strcmp(fname, FILL_END_DELIM) &&
				 point < size());
	}

	//
	// call the previous hook
	//
	fill_restore_session_hook(b);
}

REPLACE_FUNC("fill", "restore_session_hook")
