//
//	ALL NEW LINE/COL DISPLAY ROUTINES -- BY MORGWN McCARTY
//	------------------------------------------------------
//
//	Reasons for writing this stuff:
//
//		*	The built in Epsilon line number stuff for the mode line causes
//			the editor to grind to a halt once you get past the 1000 line
//			mark (at least on my machine).
//
//		*	The curspos.e routines by Jonas Hedberg had some problems when
//			deleting text which caused the row number to get off.  Also,
//			there was a long delay when jumping to the end of a large file.
//
//	Features:
//
//		*	Counts lines in the background so when you make a really big
//			jump (from line 1 to line 19438) you can start editing right
//			away.
//
//		*	Knows when text has been deleted or things have gotten off
//			and will recount lines.  A backup position is kept and if
//			it didn't also get messed up, a quick recount will be done
//			from that position.
//
//		*	Uses Epsilons "lines_between" primitive for more speed.
//
//	Notes:
//
//		Thanks to Jonas Hedberg for the original curspos.e!
//
//	Revisions:
//
//		> 12/7/94 10:02am --mm-- initial release
//

#include "eel.h"


// does user want this stuff on?
user int	want_line_col_display = 1;

// color of row and column text
color_class row_col_color =
	{ COLOR( colCYAN + colINTENSE, colBLACK ), MCNORM };

// saved position
#define		_BLOCK_SIZE 2750
buffer spot saved_spot;
buffer int	saved_point;
buffer int	saved_row;

// previous position
buffer spot prev_spot;
buffer int	prev_point;

// current row number
buffer int	row;

// flag if we're doing background line counting
buffer int	counting;


do_count_lines()
{
	// init vars for new buffer
	if ( !saved_spot )
	{
		saved_spot	= alloc_spot();
		*saved_spot = saved_point = 0;
		saved_row	= 1;
		
		prev_spot	= alloc_spot();
		*prev_spot	= prev_point = 0;
		row			= 1;

		counting	= 0;			
	}
	
	// did our position get messed up from a delete or something?
	if ( *prev_spot != prev_point )
	{
		// can we count from our saved point?
		if ( *saved_spot == saved_point )
		{
			*prev_spot	= prev_point = saved_point;
			row			= saved_row;
		}

		// or do we have to start all over from the top of the buffer?
		else
		{
			*saved_spot	= saved_point = 0;
			saved_row	= 1;
			
			*prev_spot	= prev_point = 0;
			row			= 1;
		}
	}

	// or check if at top of buffer
	else if ( !point )
	{
		*saved_spot	= saved_point = 0;
		saved_row	= 1;
		
		*prev_spot	= prev_point = 0;
		row			= 1;
	}

	// count the lines
	counting = 1;
	do_background_count_lines();
}
		 

do_background_count_lines()
{
	int count_point;					// count lines up/down to this point 
	char tmp_str[ 35 ];					// display string (a bit oversized)

	// count forwards
	if ( point > prev_point )
	{
		// only count lines contained in so big a region
		count_point = prev_point + _BLOCK_SIZE;
		
		// if point falls within block, we're done
		if ( point < count_point )
		{
			count_point = point;
			counting	= 0;
		}

		// otherwise save our backup location
		else
		{
			saved_row	= row;
			*saved_spot = saved_point = prev_point;
		}

		// count some lines
		row	+= lines_between( prev_point, count_point, 0 );
	}

	// count backwards
	else
	{
		// only count lines contained in a _BLOCK_SIZE region
		count_point = prev_point - _BLOCK_SIZE;

		// if point falls within block, we're done
		if ( point > count_point )
		{
			count_point = point;
			counting	= 0;
		}

		// count some lines
		row	-= lines_between( count_point, prev_point, 0 );
	}

	// update display if we're done
	if ( !counting )
	{
		sprintf( tmp_str, "   line: %-6d col: %d   ", row,
												current_column() + 1 );
		term_write( screen_cols - 25, screen_lines - 1, tmp_str, 25,
												COLOROF( row_col_color ), 1 );
	}

	// save our position for the next time around
	*prev_spot = prev_point = count_point;
}


new_count_lines_getkey()
{
	if ( want_line_col_display )
	{
		do_count_lines();
	}
	return count_lines_getkey();
}
REPLACE_FUNC( "count_lines", "getkey" );


new_background_count_lines_when_idle()
{
	if ( counting )
	{
		do_background_count_lines();
	}
	background_count_lines_when_idle();
}
REPLACE_FUNC( "background_count_lines", "when_idle" );
