/*
 EPSHeader

   File: spell.e
   Author: J. Kercheval
   Created: Mon, 09/16/1991  22:46:44
*/
/*
 EPSRevision History

   J. Kercheval	 Mon, 09/16/1991 22:46:52	creation
   J. Kercheval	 Sun, 09/22/1991 22:20:02	add automatic word file search and load
   J. Kercheval	 Sun, 06/21/1992 00:29:35	modify for V6.0 highlighted regions
   J. Kercheval	 Wed, 07/14/1993 12:47:24	cleanup highlight and memory allocations on abort
*/

/*
 * This file implements a basic spelling checker which relies upon an
 * external list of correctly spelled, sorted words in an ascii format and
 * delimited by newlines.  The basic algorithm is brute force and simple
 * marches through every word in the input region (or buffer) and checks that
 * word against the wordlist.  If the word in not contained in the word list
 * then an ignore buffer which is being maintained is looked at and if the
 * word is not there then the user is presented with the word and allowed to
 * change it.  Hitting just a <CR> will cause the word to be placed in the
 * ignore buffer for future occurances of that word.  Case is ignored in this
 * speller.  Possible future enhancements are to allow adding of words to the
 * dictionary and to allow saving of additional user dictionaries.  There are
 * two commands implemented, spell_region and spell_buffer (spell_buffer is
 * just a call to spell_region with the values of point and mark as 0 and
 * size() respectively.  The search of the word in the dictionary is done
 * via a binary search.  This brute force method is considerably slower than
 * a trie or dictionary method but is sufficient for a programmer's editor
 * and is quite a bit more flexible and easier to write.  A new buffer is
 * created by the name of "-words" which is not deleted upon completion to
 * allow for faster response on later spell checks.
 *
 * This code is dedicated to the public domain with the caveat that Lugaru is
 * welcome to use this within their distribution source code which is
 * supplied with Epsilon.
 *
 *		John Kercheval
 *		15563 10th Ave NE
 *		Seattle, WA 98155-6209
 *		INTERNET: johnk@wrq.com
 *		Compu$erve: 72450,3702
 *
 *
 * Load binsrch.b, then load spell.b.  Place a dictionary by the name
 * of WORDS.LST in your epsilon path (as signified by the environment
 * variable EPSPATH).  The same place as the file EDOC is a good
 * place to place the dictionary file.  Do you want to bind these
 * commands to a key (I did not want to so I left them out of the
 * function header)?  Now save the state file and you are set...
 *
 *								jbk
 */

#include <eel.h>

#ifndef BOOLEAN
#define BOOLEAN int
#define TRUE 1
#define FALSE 0
#endif

/*
 * This is the region handle used to show the spell string
 */
int spell_region_handle = 0;


/*----------------------------------------------------------------------------
 *
 * spell_region_off removes the current spell region highlight.
 *
 ---------------------------------------------------------------------------*/

spell_region_off()
{
	if (spell_region_handle) { 
		remove_region(spell_region_handle);
		spell_region_handle = 0; 
	}
}


/*----------------------------------------------------------------------------
 *
 * SpellIsSpecial() is a helper function to determine if a particular word
 * contains one of the special chars (ie. a number or underscore)
 *
 ---------------------------------------------------------------------------*/

BOOLEAN SpellIsSpecial(s)
    char *s;
{
    while (*s) {
        if (isdigit(*s))
            return TRUE;
        if (*s == '_')
            return TRUE;
        s++;
    }
    return FALSE;
}

/*----------------------------------------------------------------------------
 *
 * do_spell_region() sets up the needed buffers initializes the point in the
 * input buffer and repeatedly calls spell_check to actually perform the
 * binary search for the element
 *
 ---------------------------------------------------------------------------*/

do_spell_region(inbuf, wordbuf)
    char *inbuf;
    char *wordbuf;
{
    char elem[256];
    char res[256];

    char *ignore;

    spot end = alloc_spot();
	spot match_spot = alloc_spot();

    int region_size;

	jmp_buf *old_level = top_level;
	jmp_buf this_level;
	int ret;

    /* store the old point, mark and buffer, store and turn of highlight */
	save_spot point, mark;
	save_var bufname;
	save_var _highlight_control = 0;

    /* create ignore buffer */
    ignore = temp_buf();

	/* 
	 * remove the highlight and clean up if abort key is pressed or
	 * error occurs while processing.
	 */
	top_level = &this_level;
	if (ret = setjmp(top_level)) {
		spell_region_off();
		delete_buffer(ignore);
		free_spot(end);
		free_spot(match_spot);
		top_level = old_level;
		longjmp(top_level, ret);
	}

    /* move to buffer */
    bufname = inbuf;

    /* set point and end location */
    if (point > mark) {
        *end = point;
        point = mark;
    }
    else {
        *end = mark;
    }

    /* get the size of the region */
    region_size = *end - point;

    /* run through the region */
    while (point < *end) {

        /* place a status message */
        note("%d%% complete ...",
            (int) (100 * (region_size - (*end - point))) / region_size);

        /* handle the case where the word is not spelled correctly */
        if (re_search(1, word_pattern)) {

			/* highlight the region (and thus the word) */
			*match_spot = matchstart;
			spell_region_handle = add_region(match_spot, point_spot,
				color_class highlight, REGNORM);

            grab(matchstart, point, elem);
            bufname = wordbuf;
            if (!do_binary_search(elem)) {
                bufname = ignore;
                point = 0;
                if (!search(1, elem)) {
                    if (!SpellIsSpecial(elem)) {
                        refresh();
                        get_strdef(res, "Correct or <CR> to ignore", elem);
                        if (strfcmp(elem, res)) {

                            /* user changed the elem */
                            bufname = inbuf;
                            delete(matchstart, point);
                            stuff(res);
                            refresh();
                        }
                        else {

                            /* user ignored this word */
                            point = 0;
                            stuff(elem);
                            stuff("\n");
                        }
                    }
                }
            }

			/* we no longer need the region */
			spell_region_off();

            bufname = inbuf;
        }
    }

    /* clean up */
    note("Done.");
    delete_buffer(ignore);
    free_spot(end);
    free_spot(match_spot);

	/* deactivate highlight and restore the level */
	spell_region_off();
	top_level = old_level;
}

#define WORD_LIST_BUFFER "-words"
#define WORD_FILE_NAME "words.lst"

command spell_region()
{
    char wordbuf[40];
    char *words;
    char *oldbuf = bufname;
    int file_error;

    strcpy(wordbuf, WORD_LIST_BUFFER);

    if (!exist(wordbuf)) {

        /* obtain the word file */
        words = lookpath(WORD_FILE_NAME);
        if (*words) {
            create(wordbuf);
            bufname = wordbuf;
            file_error = file_read(words, strip_returns);
            bufname = oldbuf;
            if (file_error) {
                say("Error reading word file.");
                return;
            }
        }
        else {
            gripe("Can't find word file %s", WORD_FILE_NAME);
            return;
        }

    }

    /* run the spell checker */
    do_spell_region(bufname, wordbuf);
}

command spell_buffer()
{
    /* store the old point and mark */
	save_spot point, mark;

    /* set the region */
    mark = 0;
    point = size();

    /* run the spell checker */
    spell_region();
}
